/**
 * ZNZ dynpricing - Content Script
 * Detects page_data on e-commerce pages and shows debug link
 */

(function() {
    'use strict';

    const DEBUG_BASE_URL = 'https://admin.elpricing.com/admin/debug';
    
    // Domain to strategy_id mapping (can be extended)
    const DOMAIN_STRATEGY_MAP = {
        'casadelprofumo.it': 19,
        'parfum-zentrum.de': 1,
        'parfuemerie.de': 2,
        'beautywelt.de': 3,
        'flaconi.de': 4,
        'notino.cz': 5,
        'notino.de': 6,
        'notino.it': 7,
        'douglas.de': 8,
        'douglas.cz': 9,
        'douglas.it': 10,
        'parfumdreams.de': 11,
        'parfumdreams.it': 12,
        'easycosmeticos.com': 13
    };

    /**
     * Extract domain from current URL
     */
    function getCurrentDomain() {
        const hostname = window.location.hostname;
        // Remove www. prefix if present
        return hostname.replace(/^www\./, '');
    }

    /**
     * Get strategy ID for domain (default to 19 if not mapped)
     */
    function getStrategyId(domain) {
        // Check exact match first
        if (DOMAIN_STRATEGY_MAP[domain]) {
            return DOMAIN_STRATEGY_MAP[domain];
        }
        
        // Check if domain ends with any mapped domain
        for (const [mappedDomain, strategyId] of Object.entries(DOMAIN_STRATEGY_MAP)) {
            if (domain.endsWith(mappedDomain)) {
                return strategyId;
            }
        }
        
        return 19; // Default strategy
    }

    /**
     * Parse page_data from script tags
     */
    function findPageData() {
        const scripts = document.getElementsByTagName('script');
        
        for (const script of scripts) {
            const content = script.textContent || script.innerText;
            
            if (!content) continue;
            
            // Look for page_data assignment
            const pageDataMatch = content.match(/page_data\s*=\s*(\{[\s\S]*?\});/);
            
            if (pageDataMatch) {
                try {
                    // Try to parse the JSON
                    const pageData = JSON.parse(pageDataMatch[1]);
                    return pageData;
                } catch (e) {
                    // Try alternative parsing - sometimes JSON has issues
                    try {
                        // Use Function to evaluate (safer than eval)
                        const pageData = (new Function('return ' + pageDataMatch[1]))();
                        return pageData;
                    } catch (e2) {
                        console.log('[ZNZ dynpricing] Could not parse page_data:', e2);
                    }
                }
            }
        }
        
        // Also check if page_data exists as global variable
        if (typeof window.page_data !== 'undefined') {
            return window.page_data;
        }
        
        return null;
    }

    /**
     * Extract product information from page_data
     */
    function extractProductInfo(pageData) {
        if (!pageData) return null;
        
        const products = [];
        
        // Check products array
        if (pageData.products && Array.isArray(pageData.products)) {
            for (const product of pageData.products) {
                const productInfo = {
                    id: product.idProduct || product.id || product.productId,
                    ean: product.EAN || product.ean,
                    code: product.code || product.productCode,
                    name: product.name,
                    price: product.price || product.priceWithVat,
                    producer: product.producer || product.brand,
                    url: product.url
                };
                
                if (productInfo.id) {
                    products.push(productInfo);
                }
            }
        }
        
        // Check single product
        if (pageData.product) {
            const product = pageData.product;
            const productInfo = {
                id: product.idProduct || product.id || product.productId,
                ean: product.EAN || product.ean,
                code: product.code || product.productCode,
                name: product.name,
                price: product.price || product.priceWithVat,
                producer: product.producer || product.brand,
                url: product.url
            };
            
            if (productInfo.id) {
                products.push(productInfo);
            }
        }
        
        return products.length > 0 ? products : null;
    }

    /**
     * Create debug link URL
     */
    function createDebugUrl(domain, productId, strategyId) {
        const params = new URLSearchParams({
            domain: domain,
            product_id: productId,
            strategy_id: strategyId
        });
        return `${DEBUG_BASE_URL}?${params.toString()}`;
    }

    /**
     * Create and inject the debug widget
     */
    function createDebugWidget(products, domain, strategyId) {
        // Remove existing widget if any
        const existing = document.getElementById('elpricing-debug-widget');
        if (existing) {
            existing.remove();
        }

        // Create widget container
        const widget = document.createElement('div');
        widget.id = 'elpricing-debug-widget';
        widget.className = 'elpricing-widget';

        // Create header
        const header = document.createElement('div');
        header.className = 'elpricing-widget-header';
        header.innerHTML = `
            <span class="elpricing-logo">🔍 ZNZ dynpricing</span>
            <button class="elpricing-toggle" title="Toggle">−</button>
        `;
        widget.appendChild(header);

        // Create content
        const content = document.createElement('div');
        content.className = 'elpricing-widget-content';

        // Domain info
        const domainInfo = document.createElement('div');
        domainInfo.className = 'elpricing-domain-info';
        domainInfo.innerHTML = `<strong>Domain:</strong> ${domain}<br><strong>Strategy:</strong> ${strategyId}`;
        content.appendChild(domainInfo);

        // Products list
        const productList = document.createElement('div');
        productList.className = 'elpricing-product-list';

        for (const product of products) {
            const productItem = document.createElement('div');
            productItem.className = 'elpricing-product-item';

            const debugUrl = createDebugUrl(domain, product.id, strategyId);
            
            productItem.innerHTML = `
                <div class="elpricing-product-info">
                    <span class="elpricing-product-name" title="${product.name || 'Unknown'}">${truncate(product.name || 'Product', 30)}</span>
                    <span class="elpricing-product-details">
                        ID: ${product.id}
                        ${product.ean ? `| EAN: ${product.ean}` : ''}
                        ${product.price ? `| €${product.price}` : ''}
                    </span>
                </div>
                <a href="${debugUrl}" target="_blank" class="elpricing-debug-link" title="Open in ELPricing Debug">
                    🔧 Debug
                </a>
            `;
            
            productList.appendChild(productItem);
        }

        content.appendChild(productList);
        widget.appendChild(content);

        // Add toggle functionality
        const toggleBtn = header.querySelector('.elpricing-toggle');
        toggleBtn.addEventListener('click', () => {
            widget.classList.toggle('elpricing-collapsed');
            toggleBtn.textContent = widget.classList.contains('elpricing-collapsed') ? '+' : '−';
        });

        // Make header draggable
        makeDraggable(widget, header);

        // Inject widget
        document.body.appendChild(widget);

        console.log('[ZNZ dynpricing] Widget created with', products.length, 'product(s)');
    }

    /**
     * Truncate text
     */
    function truncate(text, maxLength) {
        if (!text) return '';
        return text.length > maxLength ? text.substring(0, maxLength) + '...' : text;
    }

    /**
     * Make element draggable
     */
    function makeDraggable(element, handle) {
        let isDragging = false;
        let startX, startY, startLeft, startTop;

        handle.addEventListener('mousedown', (e) => {
            if (e.target.tagName === 'BUTTON') return;
            
            isDragging = true;
            startX = e.clientX;
            startY = e.clientY;
            
            const rect = element.getBoundingClientRect();
            startLeft = rect.left;
            startTop = rect.top;
            
            element.style.transition = 'none';
            handle.style.cursor = 'grabbing';
        });

        document.addEventListener('mousemove', (e) => {
            if (!isDragging) return;
            
            const deltaX = e.clientX - startX;
            const deltaY = e.clientY - startY;
            
            element.style.left = (startLeft + deltaX) + 'px';
            element.style.top = (startTop + deltaY) + 'px';
            element.style.right = 'auto';
            element.style.bottom = 'auto';
        });

        document.addEventListener('mouseup', () => {
            if (isDragging) {
                isDragging = false;
                element.style.transition = '';
                handle.style.cursor = 'grab';
            }
        });
    }

    /**
     * Main initialization
     */
    function init() {
        // Wait a bit for dynamic content to load
        setTimeout(() => {
            const pageData = findPageData();
            
            if (!pageData) {
                console.log('[ZNZ dynpricing] No page_data found on this page');
                return;
            }

            const products = extractProductInfo(pageData);
            
            if (!products || products.length === 0) {
                console.log('[ZNZ dynpricing] No product information found in page_data');
                return;
            }

            const domain = getCurrentDomain();
            const strategyId = getStrategyId(domain);

            console.log('[ZNZ dynpricing] Found', products.length, 'product(s) on', domain);
            
            createDebugWidget(products, domain, strategyId);
        }, 1000);
    }

    // Run on page load
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }

    // Also run on dynamic page changes (SPA support)
    let lastUrl = location.href;
    new MutationObserver(() => {
        const url = location.href;
        if (url !== lastUrl) {
            lastUrl = url;
            setTimeout(init, 1500);
        }
    }).observe(document, { subtree: true, childList: true });

})();
